//
//  ActuarialModelTests.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import XCTest
@testable import death_app_Watch_App

final class ActuarialModelTests: XCTestCase {
    
    var actuarialModel: ActuarialModel!
    
    override func setUpWithError() throws {
        super.setUp()
        actuarialModel = ActuarialModel()
    }
    
    override func tearDownWithError() throws {
        actuarialModel = nil
        super.tearDown()
    }
    
    // MARK: - Gompertz-Makeham Function Tests
    
    func testGompertzMakehamBaselineCalculation() throws {
        let age = 40.0
        let result = actuarialModel.calculateGompertzMakeham(age: age)
        
        XCTAssertGreaterThan(result, 0, "Gompertz-Makeham result should be positive")
        XCTAssertLessThan(result, 1, "Gompertz-Makeham result should be less than 1")
    }
    
    func testGompertzMakehamAgeProgression() throws {
        let youngAge = 25.0
        let oldAge = 75.0
        
        let youngResult = actuarialModel.calculateGompertzMakeham(age: youngAge)
        let oldResult = actuarialModel.calculateGompertzMakeham(age: oldAge)
        
        XCTAssertLessThan(youngResult, oldResult, "Mortality rate should increase with age")
    }
    
    func testGompertzMakehamEdgeCases() throws {
        // Test minimum age
        let minAge = 0.0
        let minResult = actuarialModel.calculateGompertzMakeham(age: minAge)
        XCTAssertGreaterThan(minResult, 0, "Should handle minimum age")
        
        // Test maximum reasonable age
        let maxAge = 120.0
        let maxResult = actuarialModel.calculateGompertzMakeham(age: maxAge)
        XCTAssertGreaterThan(maxResult, 0, "Should handle maximum age")
        XCTAssertLessThan(maxResult, 1, "Should not exceed probability bounds")
    }
    
    // MARK: - Cox Proportional Hazards Model Tests
    
    func testCoxProportionalHazardsBaseline() throws {
        let riskFactors: [String: Double] = [
            "smoking": 0.0,
            "exercise": 1.0,
            "bmi": 22.0
        ]
        
        let hazardRatio = actuarialModel.calculateCoxHazardRatio(riskFactors: riskFactors)
        
        XCTAssertGreaterThan(hazardRatio, 0, "Hazard ratio should be positive")
        XCTAssertEqual(hazardRatio, 1.0, accuracy: 0.1, "Baseline should be close to 1.0")
    }
    
    func testCoxProportionalHazardsWithRisks() throws {
        let highRiskFactors: [String: Double] = [
            "smoking": 1.0,
            "exercise": 0.0,
            "bmi": 35.0,
            "alcohol": 1.0
        ]
        
        let lowRiskFactors: [String: Double] = [
            "smoking": 0.0,
            "exercise": 1.0,
            "bmi": 22.0,
            "alcohol": 0.0
        ]
        
        let highRiskRatio = actuarialModel.calculateCoxHazardRatio(riskFactors: highRiskFactors)
        let lowRiskRatio = actuarialModel.calculateCoxHazardRatio(riskFactors: lowRiskFactors)
        
        XCTAssertGreaterThan(highRiskRatio, lowRiskRatio, "High risk should have higher hazard ratio")
    }
    
    // MARK: - Risk Factor Calculation Tests
    
    func testBMIRiskCalculation() throws {
        let normalBMI = 22.0
        let obeseBMI = 35.0
        let underweightBMI = 16.0
        
        let normalRisk = actuarialModel.calculateBMIRisk(bmi: normalBMI)
        let obeseRisk = actuarialModel.calculateBMIRisk(bmi: obeseBMI)
        let underweightRisk = actuarialModel.calculateBMIRisk(bmi: underweightBMI)
        
        XCTAssertLessThan(normalRisk, obeseRisk, "Obesity should increase risk")
        XCTAssertLessThan(normalRisk, underweightRisk, "Underweight should increase risk")
    }
    
    func testSmokingRiskCalculation() throws {
        let nonSmokerRisk = actuarialModel.calculateSmokingRisk(smokingStatus: .never)
        let currentSmokerRisk = actuarialModel.calculateSmokingRisk(smokingStatus: .current)
        let formerSmokerRisk = actuarialModel.calculateSmokingRisk(smokingStatus: .former)
        
        XCTAssertLessThan(nonSmokerRisk, formerSmokerRisk, "Former smoking should increase risk vs never")
        XCTAssertLessThan(formerSmokerRisk, currentSmokerRisk, "Current smoking should be highest risk")
    }
    
    func testExerciseRiskCalculation() throws {
        let sedentaryRisk = actuarialModel.calculateExerciseRisk(minutesPerWeek: 0)
        let moderateRisk = actuarialModel.calculateExerciseRisk(minutesPerWeek: 150)
        let highRisk = actuarialModel.calculateExerciseRisk(minutesPerWeek: 300)
        
        XCTAssertGreaterThan(sedentaryRisk, moderateRisk, "Exercise should reduce risk")
        XCTAssertGreaterThanOrEqual(moderateRisk, highRisk, "More exercise should maintain or reduce risk")
    }
    
    // MARK: - Boundary Condition Tests
    
    func testNegativeAgeHandling() throws {
        let negativeAge = -5.0
        let result = actuarialModel.calculateLifeExpectancy(age: negativeAge, riskFactors: [:])
        
        XCTAssertNil(result, "Should return nil for negative age")
    }
    
    func testExtremeAgeHandling() throws {
        let extremeAge = 150.0
        let result = actuarialModel.calculateLifeExpectancy(age: extremeAge, riskFactors: [:])
        
        XCTAssertNotNil(result, "Should handle extreme ages gracefully")
        if let result = result {
            XCTAssertLessThan(result, extremeAge + 10, "Life expectancy should be reasonable")
        }
    }
    
    func testInvalidRiskFactorHandling() throws {
        let invalidRiskFactors: [String: Double] = [
            "bmi": -5.0,
            "exercise": -100.0,
            "unknown_factor": 999.0
        ]
        
        let result = actuarialModel.calculateLifeExpectancy(age: 40, riskFactors: invalidRiskFactors)
        XCTAssertNotNil(result, "Should handle invalid risk factors gracefully")
    }
    
    // MARK: - Statistical Accuracy Tests
    
    func testLifeExpectancyConsistency() throws {
        let age = 45.0
        let riskFactors: [String: Double] = ["bmi": 24.0, "exercise": 200.0]
        
        var results: [Double] = []
        for _ in 0..<10 {
            if let result = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors) {
                results.append(result)
            }
        }
        
        XCTAssertEqual(results.count, 10, "Should consistently return results")
        
        // Check that results are consistent (within reasonable variance)
        let average = results.reduce(0, +) / Double(results.count)
        let variance = results.map { pow($0 - average, 2) }.reduce(0, +) / Double(results.count)
        
        XCTAssertLessThan(variance, 1.0, "Results should be consistent with low variance")
    }
    
    func testGenderDifferenceHandling() throws {
        let age = 50.0
        let riskFactors: [String: Double] = [:]
        
        let maleResult = actuarialModel.calculateLifeExpectancy(age: age, gender: .male, riskFactors: riskFactors)
        let femaleResult = actuarialModel.calculateLifeExpectancy(age: age, gender: .female, riskFactors: riskFactors)
        
        XCTAssertNotNil(maleResult)
        XCTAssertNotNil(femaleResult)
        
        if let male = maleResult, let female = femaleResult {
            // Generally, females have slightly higher life expectancy
            XCTAssertLessThanOrEqual(male, female + 5, "Gender differences should be within expected range")
        }
    }
    
    // MARK: - Performance Tests
    
    func testCalculationPerformance() throws {
        let age = 40.0
        let riskFactors: [String: Double] = [
            "bmi": 25.0,
            "smoking": 0.0,
            "exercise": 180.0,
            "alcohol": 0.5
        ]
        
        measure {
            for _ in 0..<1000 {
                _ = actuarialModel.calculateLifeExpectancy(age: age, riskFactors: riskFactors)
            }
        }
    }
    
    func testConcurrentCalculations() throws {
        let expectation = XCTestExpectation(description: "Concurrent calculations")
        let dispatchGroup = DispatchGroup()
        var results: [Double] = []
        let resultsLock = NSLock()
        
        for i in 20...80 {
            dispatchGroup.enter()
            DispatchQueue.global().async {
                if let result = self.actuarialModel.calculateLifeExpectancy(age: Double(i), riskFactors: [:]) {
                    resultsLock.lock()
                    results.append(result)
                    resultsLock.unlock()
                }
                dispatchGroup.leave()
            }
        }
        
        dispatchGroup.notify(queue: .main) {
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 5.0)
        XCTAssertEqual(results.count, 61, "Should handle concurrent calculations")
    }
}